<?php
@session_start();

$page_title = "Admin Login - SavingVest Mart";

// If admin is already logged in, redirect to dashboard
if (!empty($_SESSION['admin_logged_in'])) {
    header('Location: index.php');
    exit();
}

$error_message = '';

// Include database connection helper
require_once '../config/database.php';

require_once 'db.php';

// Handle login form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Basic input handling
    $email = trim($_POST['email'] ?? '');
    $password = $_POST['password'] ?? '';

    if ($email === '' || $password === '') {
        $error_message = 'Please fill in all fields.';
    } else {
        // Get a mysqli connection from your helper
        $conn = null;
        if (function_exists('getDBConnection')) {
            $conn = getDBConnection();
        }

        if (!($conn instanceof mysqli)) {
            $error_message = 'Database connection not available.';
        } else {
            // Prepare the admin lookup
            $sql = "SELECT id, email, password, first_name, last_name, role, is_active
                    FROM users
                    WHERE email = ? AND role = 'admin'
                    LIMIT 1";

            if ($stmt = $conn->prepare($sql)) {
                $stmt->bind_param("s", $email);
                $stmt->execute();
                $result = $stmt->get_result();
                $admin  = $result ? $result->fetch_assoc() : null;
                $stmt->close();

                if ($admin && (int)$admin['is_active'] === 1 && password_verify($password, $admin['password'])) {
                    // Successful login — harden the session
                    session_regenerate_id(true);

                    // Build a display name from your columns
                    $admin_name = trim(($admin['first_name'] ?? '') . ' ' . ($admin['last_name'] ?? ''));

                    // Set admin session values
                    $_SESSION['admin_logged_in'] = true;
                    $_SESSION['admin_id']        = (int)$admin['id'];
                    $_SESSION['admin_email']     = $admin['email'];
                    $_SESSION['admin_name']      = $admin_name !== '' ? $admin_name : $admin['email'];
                    $_SESSION['admin_login_time']= time();

                    // Optionally update last_login if the column exists
                    $colCheck = $conn->query("SHOW COLUMNS FROM `users` LIKE 'last_login'");
                    if ($colCheck && $colCheck->num_rows === 1) {
                        if ($upd = $conn->prepare("UPDATE users SET last_login = NOW() WHERE id = ?")) {
                            $upd->bind_param("i", $_SESSION['admin_id']);
                            $upd->execute();
                            $upd->close();
                        }
                    }

                    $conn->close();
                    header('Location: index.php');
                    exit();
                } else {
                    // Distinguish deactivated vs invalid creds
                    if ($admin && (int)$admin['is_active'] !== 1) {
                        $error_message = 'Your admin account is deactivated. Please contact support.';
                    } else {
                        $error_message = 'Invalid email or password.';
                    }
                }
            } else {
                // Prepare failed — show DB error safely (during development)
                $error_message = 'Database error (prepare): ' . $conn->error;
            }

            $conn->close();
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8" />
    <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
    <title><?php echo htmlspecialchars($page_title, ENT_QUOTES, 'UTF-8'); ?></title>
    <script src="https://cdn.tailwindcss.com"></script>
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        primary: {
                            50: '#f0f9ff',
                            100: '#e0f2fe',
                            200: '#bae6fd',
                            300: '#7dd3fc',
                            400: '#38bdf8',
                            500: '#0891b2',
                            600: '#0e7490',
                            700: '#0f766e',
                            800: '#115e59',
                            900: '#134e4a'
                        }
                    }
                }
            }
        }
    </script>
</head>
<body class="bg-gray-100 min-h-screen flex items-center justify-center">
    <div class="max-w-md w-full mx-4">
        <div class="bg-white rounded-2xl shadow-xl p-8">
            <!-- Logo and Header -->
            <div class="text-center mb-8">
                <div class="flex items-center justify-center space-x-3 mb-4">
                    <div class="w-12 h-12 bg-primary-500 rounded-xl flex items-center justify-center">
                        <svg class="w-6 h-6 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                  d="M12 15v2m-6 4h12a2 2 0 002-2v-6a2 2 0 00-2-2H6a2 2 0 00-2 2v6a2 2 0 002 2zm10-10V7a4 4 0 00-8 0v4h8z"></path>
                        </svg>
                    </div>
                    <span class="text-2xl font-bold text-gray-900">Admin Portal</span>
                </div>
                <h1 class="text-2xl font-bold text-gray-900 mb-2">Welcome Back</h1>
                <p class="text-gray-600">Sign in to your admin account</p>
            </div>

            <!-- Error Message -->
            <?php if (!empty($error_message)): ?>
                <div class="bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded-lg mb-6">
                    <?php echo htmlspecialchars($error_message, ENT_QUOTES, 'UTF-8'); ?>
                </div>
            <?php endif; ?>

            <!-- Login Form -->
            <form method="POST" class="space-y-6" novalidate>
                <div>
                    <label for="email" class="block text-sm font-medium text-gray-700 mb-2">Email Address</label>
                    <input
                        type="email"
                        id="email"
                        name="email"
                        value="<?php echo htmlspecialchars($_POST['email'] ?? '', ENT_QUOTES, 'UTF-8'); ?>"
                        class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-transparent transition-colors"
                        placeholder="admin@savingvest.com"
                        required
                    />
                </div>

                <div>
                    <label for="password" class="block text-sm font-medium text-gray-700 mb-2">Password</label>
                    <input
                        type="password"
                        id="password"
                        name="password"
                        class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-transparent transition-colors"
                        placeholder="Enter your password"
                        required
                    />
                </div>

                <button
                    type="submit"
                    class="w-full bg-primary-500 text-white py-3 px-4 rounded-lg hover:bg-primary-600 focus:ring-2 focus:ring-primary-500 focus:ring-offset-2 font-medium transition-colors"
                >
                    Sign In
                </button>
            </form>

            <!-- Demo Credentials -->
            <div class="mt-6 p-4 bg-gray-50 rounded-lg">
                <p class="text-sm text-gray-600 mb-2"><strong>Demo Admin Credentials:</strong></p>
                <p class="text-sm text-gray-600">Email: admin@savingvest.com</p>
                <p class="text-sm text-gray-600">Password: password123</p>
            </div>

            <!-- Back to Site -->
            <div class="mt-6 text-center">
                <a href="../index.php" class="text-primary-600 hover:text-primary-700 text-sm font-medium">
                    ← Back to Main Site
                </a>
            </div>
        </div>
    </div>
</body>
</html>
