<?php
session_start();
require_once '../config/database.php';

header('Content-Type: application/json');

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    echo json_encode(['success' => false, 'message' => 'Please log in']);
    exit;
}

// Get JSON input
$input = json_decode(file_get_contents('php://input'), true);

if (!isset($input['cart_item_id']) || !isset($input['quantity'])) {
    echo json_encode(['success' => false, 'message' => 'Missing required parameters']);
    exit;
}

$user_id = $_SESSION['user_id'];
$cart_item_id = intval($input['cart_item_id']);
$quantity = intval($input['quantity']);

if ($quantity < 1) {
    echo json_encode(['success' => false, 'message' => 'Invalid quantity']);
    exit;
}

$conn = getDBConnection();

// Verify cart item belongs to user and get product info
$verify_sql = "SELECT ci.product_id, p.price, p.sale_price, p.stock_quantity 
               FROM cart_items ci 
               JOIN products p ON ci.product_id = p.id 
               WHERE ci.id = ? AND ci.user_id = ?";
$verify_stmt = $conn->prepare($verify_sql);
$verify_stmt->bind_param("ii", $cart_item_id, $user_id);
$verify_stmt->execute();
$cart_item = $verify_stmt->get_result()->fetch_assoc();

if (!$cart_item) {
    echo json_encode(['success' => false, 'message' => 'Cart item not found']);
    exit;
}

if ($quantity > $cart_item['stock_quantity']) {
    echo json_encode(['success' => false, 'message' => 'Not enough stock available']);
    exit;
}

// Update cart item
$update_sql = "UPDATE cart_items SET quantity = ?, updated_at = CURRENT_TIMESTAMP WHERE id = ?";
$update_stmt = $conn->prepare($update_sql);
$update_stmt->bind_param("ii", $quantity, $cart_item_id);

if ($update_stmt->execute()) {
    // Calculate new item total
    $item_price = $cart_item['sale_price'] ?: $cart_item['price'];
    $item_total = $item_price * $quantity;
    
    // Get updated cart totals
    $totals_sql = "SELECT 
                    SUM(COALESCE(p.sale_price, p.price) * ci.quantity) as subtotal,
                    SUM(ci.quantity) as total_items
                   FROM cart_items ci
                   JOIN products p ON ci.product_id = p.id
                   WHERE ci.user_id = ?";
    $totals_stmt = $conn->prepare($totals_sql);
    $totals_stmt->bind_param("i", $user_id);
    $totals_stmt->execute();
    $totals = $totals_stmt->get_result()->fetch_assoc();
    
    $subtotal = floatval($totals['subtotal']);
    $shipping = $subtotal >= 99 ? 0 : 9.99;
    $tax = $subtotal * 0.08;
    $total = $subtotal + $shipping + $tax;
    
    echo json_encode([
        'success' => true,
        'item_total' => number_format($item_total, 2),
        'cart_totals' => [
            'subtotal' => number_format($subtotal, 2),
            'shipping' => number_format($shipping, 2),
            'tax' => number_format($tax, 2),
            'total' => number_format($total, 2),
            'total_items' => intval($totals['total_items'])
        ]
    ]);
} else {
    echo json_encode(['success' => false, 'message' => 'Failed to update cart']);
}

$conn->close();
?>
