// Main JavaScript functionality for SavingVest Mart

// Mobile menu toggle
function toggleMobileMenu() {
  const mobileMenu = document.getElementById("mobile-menu")
  mobileMenu.classList.toggle("hidden")
}

// Add to cart functionality
function addToCart(productId, quantity = 1) {
  // Check if user is logged in
  fetch("api/check_auth.php")
    .then((response) => response.json())
    .then((data) => {
      if (!data.authenticated) {
        window.location.href = "login.php?redirect=" + encodeURIComponent(window.location.href)
        return
      }

      // Add to cart
      return fetch("api/add_to_cart.php", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({
          product_id: productId,
          quantity: quantity,
        }),
      })
    })
    .then((response) => response.json())
    .then((data) => {
      if (data.success) {
        showNotification("Product added to cart!", "success")
        updateCartCount()
      } else {
        showNotification(data.message || "Failed to add product to cart", "error")
      }
    })
    .catch((error) => {
      console.error("Error:", error)
      showNotification("An error occurred. Please try again.", "error")
    })
}

// Update cart count in header
function updateCartCount() {
  fetch("api/get_cart_count.php")
    .then((response) => response.json())
    .then((data) => {
      const cartBadge = document.querySelector(".cart-count")
      if (cartBadge) {
        if (data.count > 0) {
          cartBadge.textContent = data.count
          cartBadge.classList.remove("hidden")
        } else {
          cartBadge.classList.add("hidden")
        }
      }
    })
    .catch((error) => console.error("Error updating cart count:", error))
}

// Show notification
function showNotification(message, type = "info") {
  // Create notification element
  const notification = document.createElement("div")
  notification.className = `fixed top-4 right-4 z-50 p-4 rounded-lg shadow-lg max-w-sm transform transition-all duration-300 translate-x-full`

  // Set colors based on type
  const colors = {
    success: "bg-green-500 text-white",
    error: "bg-red-500 text-white",
    info: "bg-blue-500 text-white",
    warning: "bg-yellow-500 text-black",
  }

  notification.className += ` ${colors[type] || colors.info}`
  notification.innerHTML = `
        <div class="flex items-center justify-between">
            <span>${message}</span>
            <button onclick="this.parentElement.parentElement.remove()" class="ml-4 text-lg font-bold opacity-70 hover:opacity-100">×</button>
        </div>
    `

  document.body.appendChild(notification)

  // Animate in
  setTimeout(() => {
    notification.classList.remove("translate-x-full")
  }, 100)

  // Auto remove after 5 seconds
  setTimeout(() => {
    notification.classList.add("translate-x-full")
    setTimeout(() => {
      if (notification.parentElement) {
        notification.remove()
      }
    }, 300)
  }, 5000)
}

// Search functionality
function performSearch(query) {
  if (query.trim()) {
    window.location.href = `search.php?q=${encodeURIComponent(query.trim())}`
  }
}

// Initialize on page load
document.addEventListener("DOMContentLoaded", () => {
  // Update cart count on page load
  updateCartCount()

  // Handle search form submission
  const searchForm = document.querySelector('form[action="search.php"]')
  if (searchForm) {
    searchForm.addEventListener("submit", function (e) {
      e.preventDefault()
      const query = this.querySelector('input[name="q"]').value
      performSearch(query)
    })
  }

  // Handle mobile menu clicks outside
  document.addEventListener("click", (e) => {
    const mobileMenu = document.getElementById("mobile-menu")
    const menuButton = document.querySelector('[onclick="toggleMobileMenu()"]')

    if (!mobileMenu.contains(e.target) && !menuButton.contains(e.target)) {
      mobileMenu.classList.add("hidden")
    }
  })

  // Smooth scrolling for anchor links
  document.querySelectorAll('a[href^="#"]').forEach((anchor) => {
    anchor.addEventListener("click", function (e) {
      e.preventDefault()
      const target = document.querySelector(this.getAttribute("href"))
      if (target) {
        target.scrollIntoView({
          behavior: "smooth",
          block: "start",
        })
      }
    })
  })
})

// Utility functions
function formatPrice(price) {
  return new Intl.NumberFormat("en-US", {
    style: "currency",
    currency: "USD",
  }).format(price)
}

function debounce(func, wait) {
  let timeout
  return function executedFunction(...args) {
    const later = () => {
      clearTimeout(timeout)
      func(...args)
    }
    clearTimeout(timeout)
    timeout = setTimeout(later, wait)
  }
}
