"use client"

import type React from "react"

import { createContext, useContext, useReducer, useEffect, type ReactNode } from "react"

export interface User {
  id: number
  email: string
  firstName: string
  lastName: string
  avatar?: string
  createdAt: string
}

interface AuthState {
  user: User | null
  isLoading: boolean
  isAuthenticated: boolean
}

type AuthAction =
  | { type: "LOGIN_START" }
  | { type: "LOGIN_SUCCESS"; payload: User }
  | { type: "LOGIN_FAILURE" }
  | { type: "LOGOUT" }
  | { type: "REGISTER_SUCCESS"; payload: User }
  | { type: "SET_LOADING"; payload: boolean }

const AuthContext = createContext<{
  state: AuthState
  dispatch: React.Dispatch<AuthAction>
  login: (email: string, password: string) => Promise<boolean>
  register: (email: string, password: string, firstName: string, lastName: string) => Promise<boolean>
  logout: () => void
} | null>(null)

// Mock users database
const mockUsers: (User & { password: string })[] = [
  {
    id: 1,
    email: "john@example.com",
    password: "password123",
    firstName: "John",
    lastName: "Doe",
    avatar: "/placeholder.svg?height=40&width=40&text=JD",
    createdAt: "2024-01-15",
  },
  {
    id: 2,
    email: "sarah@example.com",
    password: "password123",
    firstName: "Sarah",
    lastName: "Johnson",
    avatar: "/placeholder.svg?height=40&width=40&text=SJ",
    createdAt: "2024-02-20",
  },
]

const authReducer = (state: AuthState, action: AuthAction): AuthState => {
  switch (action.type) {
    case "LOGIN_START":
      return { ...state, isLoading: true }

    case "LOGIN_SUCCESS":
      return {
        user: action.payload,
        isLoading: false,
        isAuthenticated: true,
      }

    case "LOGIN_FAILURE":
      return {
        user: null,
        isLoading: false,
        isAuthenticated: false,
      }

    case "REGISTER_SUCCESS":
      return {
        user: action.payload,
        isLoading: false,
        isAuthenticated: true,
      }

    case "LOGOUT":
      return {
        user: null,
        isLoading: false,
        isAuthenticated: false,
      }

    case "SET_LOADING":
      return { ...state, isLoading: action.payload }

    default:
      return state
  }
}

export function AuthProvider({ children }: { children: ReactNode }) {
  const [state, dispatch] = useReducer(authReducer, {
    user: null,
    isLoading: true,
    isAuthenticated: false,
  })

  // Check for existing session on mount
  useEffect(() => {
    const checkSession = () => {
      const savedUser = localStorage.getItem("user")
      if (savedUser) {
        try {
          const user = JSON.parse(savedUser)
          dispatch({ type: "LOGIN_SUCCESS", payload: user })
        } catch (error) {
          localStorage.removeItem("user")
        }
      }
      dispatch({ type: "SET_LOADING", payload: false })
    }

    checkSession()
  }, [])

  const login = async (email: string, password: string): Promise<boolean> => {
    dispatch({ type: "LOGIN_START" })

    // Simulate API call delay
    await new Promise((resolve) => setTimeout(resolve, 1000))

    const user = mockUsers.find((u) => u.email === email && u.password === password)

    if (user) {
      const { password: _, ...userWithoutPassword } = user
      dispatch({ type: "LOGIN_SUCCESS", payload: userWithoutPassword })
      localStorage.setItem("user", JSON.stringify(userWithoutPassword))
      return true
    } else {
      dispatch({ type: "LOGIN_FAILURE" })
      return false
    }
  }

  const register = async (email: string, password: string, firstName: string, lastName: string): Promise<boolean> => {
    dispatch({ type: "LOGIN_START" })

    // Simulate API call delay
    await new Promise((resolve) => setTimeout(resolve, 1000))

    // Check if user already exists
    const existingUser = mockUsers.find((u) => u.email === email)
    if (existingUser) {
      dispatch({ type: "LOGIN_FAILURE" })
      return false
    }

    // Create new user
    const newUser: User = {
      id: mockUsers.length + 1,
      email,
      firstName,
      lastName,
      avatar: `/placeholder.svg?height=40&width=40&text=${firstName[0]}${lastName[0]}`,
      createdAt: new Date().toISOString().split("T")[0],
    }

    // Add to mock database
    mockUsers.push({ ...newUser, password })

    dispatch({ type: "REGISTER_SUCCESS", payload: newUser })
    localStorage.setItem("user", JSON.stringify(newUser))
    return true
  }

  const logout = () => {
    dispatch({ type: "LOGOUT" })
    localStorage.removeItem("user")
  }

  return <AuthContext.Provider value={{ state, dispatch, login, register, logout }}>{children}</AuthContext.Provider>
}

export function useAuth() {
  const context = useContext(AuthContext)
  if (!context) {
    throw new Error("useAuth must be used within an AuthProvider")
  }
  return context
}
