<?php
require_once 'config/database.php';

// Get product ID from URL
$product_id = isset($_GET['id']) ? intval($_GET['id']) : 0;

if (!$product_id) {
    header("Location: products.php");
    exit;
}

$conn = getDBConnection();

// Get product details
$sql = "SELECT p.*, c.name as category_name, c.slug as category_slug
        FROM products p 
        LEFT JOIN categories c ON p.category_id = c.id 
        WHERE p.id = ? AND p.is_active = 1";

$stmt = $conn->prepare($sql);
$stmt->bind_param("i", $product_id);
$stmt->execute();
$product = $stmt->get_result()->fetch_assoc();

if (!$product) {
    header("Location: products.php");
    exit;
}

// Set page meta data
$page_title = htmlspecialchars($product['name']);
$page_description = htmlspecialchars($product['short_description']);

include 'includes/header.php';

// Get product reviews - Fixed column names and table structure
$reviews_sql = "SELECT pr.*, u.first_name, u.last_name 
                FROM product_reviews pr 
                LEFT JOIN users u ON pr.user_id = u.id 
                WHERE pr.product_id = ? 
                ORDER BY pr.created_at DESC 
                LIMIT 10";

$reviews_stmt = $conn->prepare($reviews_sql);
$reviews_stmt->bind_param("i", $product_id);
$reviews_stmt->execute();
$reviews = $reviews_stmt->get_result()->fetch_all(MYSQLI_ASSOC);

// Get related products (same category, excluding current product)
$related_sql = "SELECT * FROM products 
                WHERE category_id = ? AND id != ? AND is_active = 1 
                ORDER BY average_rating DESC, review_count DESC 
                LIMIT 4";

$related_stmt = $conn->prepare($related_sql);
$related_stmt->bind_param("ii", $product['category_id'], $product_id);
$related_stmt->execute();
$related_products = $related_stmt->get_result()->fetch_all(MYSQLI_ASSOC);

$conn->close();

// Parse JSON fields
$gallery_images = json_decode($product['gallery_images'] ?? '[]', true) ?: [];
$specifications = json_decode($product['specifications'] ?? '{}', true) ?: [];
$features = json_decode($product['features'] ?? '[]', true) ?: [];

// Calculate discount percentage
$discount_percentage = 0;
if (!empty($product['sale_price']) && $product['price'] > $product['sale_price']) {
    $discount_percentage = round((($product['price'] - $product['sale_price']) / $product['price']) * 100);
}
?>

<!-- Breadcrumb -->
<nav class="bg-gray-50 py-4">
    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <div class="flex items-center space-x-2 text-sm">
            <a href="index.php" class="text-gray-600 hover:text-primary-600">Home</a>
            <svg class="w-4 h-4 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7"></path>
            </svg>
            <a href="products.php" class="text-gray-600 hover:text-primary-600">Products</a>
            <svg class="w-4 h-4 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7"></path>
            </svg>
            <a href="products.php?category=<?php echo urlencode($product['category_slug']); ?>" class="text-gray-600 hover:text-primary-600">
                <?php echo htmlspecialchars($product['category_name']); ?>
            </a>
            <svg class="w-4 h-4 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7"></path>
            </svg>
            <span class="text-gray-900 font-medium"><?php echo htmlspecialchars($product['name']); ?></span>
        </div>
    </div>
</nav>

<!-- Product Details -->
<section class="py-8">
    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <div class="grid grid-cols-1 lg:grid-cols-2 gap-12">
            <!-- Product Images -->
            <div>
                <!-- Main Image -->
                <div class="aspect-w-1 aspect-h-1 bg-gray-100 rounded-lg overflow-hidden mb-4">
                    <img id="main-image" 
                         src="<?php echo htmlspecialchars($product['image_url']); ?>" 
                         alt="<?php echo htmlspecialchars($product['name']); ?>"
                         class="w-full h-96 object-cover">
                </div>
                
                <!-- Image Gallery -->
                <?php if (!empty($gallery_images)): ?>
                <div class="grid grid-cols-4 gap-2">
                    <button onclick="changeMainImage('<?php echo htmlspecialchars($product['image_url']); ?>')" 
                            class="aspect-w-1 aspect-h-1 bg-gray-100 rounded-md overflow-hidden border-2 border-primary-500">
                        <img src="<?php echo htmlspecialchars($product['image_url']); ?>" 
                             alt="<?php echo htmlspecialchars($product['name']); ?>"
                             class="w-full h-20 object-cover">
                    </button>
                    <?php foreach ($gallery_images as $image): ?>
                    <button onclick="changeMainImage('<?php echo htmlspecialchars($image); ?>')" 
                            class="aspect-w-1 aspect-h-1 bg-gray-100 rounded-md overflow-hidden border-2 border-transparent hover:border-primary-300">
                        <img src="<?php echo htmlspecialchars($image); ?>" 
                             alt="<?php echo htmlspecialchars($product['name']); ?>"
                             class="w-full h-20 object-cover">
                    </button>
                    <?php endforeach; ?>
                </div>
                <?php endif; ?>
            </div>
            
            <!-- Product Info -->
            <div>
                <!-- Product Title and Brand -->
                <div class="mb-4">
                    <h1 class="text-3xl font-bold text-gray-900 mb-2"><?php echo htmlspecialchars($product['name']); ?></h1>
                    <p class="text-lg text-gray-600">by <?php echo htmlspecialchars($product['brand']); ?></p>
                </div>
                
                <!-- Rating and Reviews -->
                <div class="flex items-center mb-6">
                    <div class="flex items-center">
                        <?php for ($i = 1; $i <= 5; $i++): ?>
                            <svg class="w-5 h-5 <?php echo $i <= floor($product['average_rating'] ?? 0) ? 'text-yellow-400' : 'text-gray-300'; ?>" 
                                 fill="currentColor" viewBox="0 0 20 20">
                                <path d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.921-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.197-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z"></path>
                            </svg>
                        <?php endfor; ?>
                        <span class="ml-2 text-lg text-gray-600"><?php echo number_format($product['average_rating'] ?? 0, 1); ?></span>
                        <span class="ml-2 text-gray-600">(<?php echo $product['review_count'] ?? 0; ?> reviews)</span>
                    </div>
                </div>
                
                <!-- Price -->
                <div class="mb-6">
                    <div class="flex items-center space-x-4">
                        <?php if (!empty($product['sale_price'])): ?>
                            <span class="text-4xl font-bold text-primary-600">₦<?php echo number_format($product['sale_price'], 0); ?></span>
                            <span class="text-2xl text-gray-500 line-through">₦<?php echo number_format($product['price'], 0); ?></span>
                            <?php if ($discount_percentage > 0): ?>
                                <span class="bg-red-100 text-red-800 px-3 py-1 rounded-full text-sm font-medium">
                                    <?php echo $discount_percentage; ?>% OFF
                                </span>
                            <?php endif; ?>
                        <?php else: ?>
                            <span class="text-4xl font-bold text-gray-900">₦<?php echo number_format($product['price'], 0); ?></span>
                        <?php endif; ?>
                    </div>
                </div>
                
                <!-- Short Description -->
                <div class="mb-6">
                    <p class="text-lg text-gray-700 leading-relaxed"><?php echo htmlspecialchars($product['short_description']); ?></p>
                </div>
                
                <!-- Stock Status -->
                <div class="mb-6">
                    <?php if ($product['quantity'] <= 0): ?>
                        <div class="flex items-center text-red-600">
                            <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.964-.833-2.732 0L4.082 16.5c-.77.833.192 2.5 1.732 2.5z"></path>
                            </svg>
                            <span class="font-medium">Out of Stock</span>
                        </div>
                    <?php elseif ($product['quantity'] <= 5): ?>
                        <div class="flex items-center text-orange-600">
                            <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.964-.833-2.732 0L4.082 16.5c-.77.833.192 2.5 1.732 2.5z"></path>
                            </svg>
                            <span class="font-medium">Only <?php echo $product['quantity']; ?> left in stock!</span>
                        </div>
                    <?php else: ?>
                        <div class="flex items-center text-green-600">
                            <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"></path>
                            </svg>
                            <span class="font-medium">In Stock (<?php echo $product['quantity']; ?> available)</span>
                        </div>
                    <?php endif; ?>
                </div>
                
                <!-- Add to Cart Form -->
                <form class="mb-8" onsubmit="handleAddToCart(event)">
                    <div class="flex items-center space-x-4 mb-4">
                        <label for="quantity" class="text-sm font-medium text-gray-700">Quantity:</label>
                        <select id="quantity" name="quantity" 
                                class="border border-gray-300 rounded-md px-3 py-2 focus:ring-2 focus:ring-primary-500 focus:border-transparent"
                                <?php echo $product['quantity'] <= 0 ? 'disabled' : ''; ?>>
                            <?php for ($i = 1; $i <= min(10, $product['quantity']); $i++): ?>
                                <option value="<?php echo $i; ?>"><?php echo $i; ?></option>
                            <?php endfor; ?>
                        </select>
                    </div>
                    
                    <div class="flex space-x-4">
                        <button type="submit" 
                                class="flex-1 bg-primary-500 text-white py-3 px-6 rounded-lg font-semibold hover:bg-primary-600 disabled:bg-gray-400 disabled:cursor-not-allowed"
                                <?php echo $product['quantity'] <= 0 ? 'disabled' : ''; ?>>
                            <?php echo $product['quantity'] <= 0 ? 'Out of Stock' : 'Add to Cart'; ?>
                        </button>
                        <button type="button" onclick="addToWishlist(<?php echo $product['id']; ?>)"
                                class="border-2 border-gray-300 text-gray-700 py-3 px-6 rounded-lg font-semibold hover:border-primary-500 hover:text-primary-600">
                            <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4.318 6.318a4.5 4.5 0 000 6.364L12 20.364l7.682-7.682a4.5 4.5 0 00-6.364-6.364L12 7.636l-1.318-1.318a4.5 4.5 0 00-6.364 0z"></path>
                            </svg>
                        </button>
                    </div>
                </form>
                
                <!-- Product Features -->
                <?php if (!empty($features)): ?>
                <div class="mb-8">
                    <h3 class="text-lg font-semibold text-gray-900 mb-4">Key Features</h3>
                    <ul class="space-y-2">
                        <?php foreach ($features as $feature): ?>
                        <li class="flex items-start">
                            <svg class="w-5 h-5 text-primary-500 mr-2 mt-0.5 flex-shrink-0" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"></path>
                            </svg>
                            <span class="text-gray-700"><?php echo htmlspecialchars($feature); ?></span>
                        </li>
                        <?php endforeach; ?>
                    </ul>
                </div>
                <?php endif; ?>
                
                <!-- Shipping Info -->
                <div class="border-t border-gray-200 pt-6">
                    <div class="grid grid-cols-1 md:grid-cols-3 gap-4 text-sm">
                        <div class="flex items-center">
                            <svg class="w-5 h-5 text-primary-500 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M20 7l-8-4-8 4m16 0l-8 4m8-4v10l-8 4m0-10L4 7m8 4v10M4 7v10l8 4"></path>
                            </svg>
                            <span class="text-gray-700">Free shipping on orders over ₦79,200</span>
                        </div>
                        <div class="flex items-center">
                            <svg class="w-5 h-5 text-primary-500 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m5.618-4.016A11.955 11.955 0 0112 2.944a11.955 11.955 0 01-8.618 3.04A12.02 12.02 0 003 9c0 5.591 3.824 10.29 9 11.622 5.176-1.332 9-6.03 9-11.622 0-1.042-.133-2.052-.382-3.016z"></path>
                            </svg>
                            <span class="text-gray-700">1-year warranty included</span>
                        </div>
                        <div class="flex items-center">
                            <svg class="w-5 h-5 text-primary-500 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 10h10a8 8 0 018 8v2M3 10l6 6m-6-6l6-6"></path>
                            </svg>
                            <span class="text-gray-700">30-day return policy</span>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</section>

<!-- Product Details Tabs -->
<section class="py-8 bg-gray-50">
    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <div class="bg-white rounded-lg shadow-sm">
            <!-- Tab Navigation -->
            <div class="border-b border-gray-200">
                <nav class="flex space-x-8 px-6">
                    <button onclick="showTab('description')" 
                            class="tab-button py-4 px-1 border-b-2 font-medium text-sm border-primary-500 text-primary-600" 
                            id="description-tab">
                        Description
                    </button>
                    <?php if (!empty($specifications)): ?>
                    <button onclick="showTab('specifications')" 
                            class="tab-button py-4 px-1 border-b-2 font-medium text-sm border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300" 
                            id="specifications-tab">
                        Specifications
                    </button>
                    <?php endif; ?>
                    <button onclick="showTab('reviews')" 
                            class="tab-button py-4 px-1 border-b-2 font-medium text-sm border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300" 
                            id="reviews-tab">
                        Reviews (<?php echo count($reviews); ?>)
                    </button>
                </nav>
            </div>
            
            <!-- Tab Content -->
            <div class="p-6">
                <!-- Description Tab -->
                <div id="description-content" class="tab-content">
                    <div class="prose max-w-none">
                        <p class="text-gray-700 leading-relaxed text-lg">
                            <?php echo nl2br(htmlspecialchars($product['description'])); ?>
                        </p>
                    </div>
                </div>
                
                <!-- Specifications Tab -->
                <?php if (!empty($specifications)): ?>
                <div id="specifications-content" class="tab-content hidden">
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <?php foreach ($specifications as $spec_name => $spec_value): ?>
                        <div class="flex justify-between py-2 border-b border-gray-200">
                            <span class="font-medium text-gray-900"><?php echo htmlspecialchars($spec_name); ?></span>
                            <span class="text-gray-700"><?php echo htmlspecialchars($spec_value); ?></span>
                        </div>
                        <?php endforeach; ?>
                    </div>
                </div>
                <?php endif; ?>
                
                <!-- Reviews Tab -->
                <div id="reviews-content" class="tab-content hidden">
                    <?php if (empty($reviews)): ?>
                        <div class="text-center py-8">
                            <svg class="w-12 h-12 text-gray-400 mx-auto mb-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 12h.01M12 12h.01M16 12h.01M21 12c0 4.418-4.03 8-9 8a9.863 9.863 0 01-4.255-.949L3 20l1.395-3.72C3.512 15.042 3 13.574 3 12c0-4.418 4.03-8 9-8s9 3.582 9 8z"></path>
                            </svg>
                            <h3 class="text-lg font-medium text-gray-900 mb-2">No reviews yet</h3>
                            <p class="text-gray-600">Be the first to review this product!</p>
                        </div>
                    <?php else: ?>
                        <div class="space-y-6">
                            <?php foreach ($reviews as $review): ?>
                            <div class="border-b border-gray-200 pb-6">
                                <div class="flex items-center justify-between mb-2">
                                    <div class="flex items-center">
                                        <div class="flex items-center">
                                            <?php for ($i = 1; $i <= 5; $i++): ?>
                                                <svg class="w-4 h-4 <?php echo $i <= $review['rating'] ? 'text-yellow-400' : 'text-gray-300'; ?>" 
                                                     fill="currentColor" viewBox="0 0 20 20">
                                                    <path d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.921-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.197-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z"></path>
                                                </svg>
                                            <?php endfor; ?>
                                        </div>
                                        <span class="ml-2 font-medium text-gray-900">
                                            <?php echo htmlspecialchars($review['first_name'] . ' ' . substr($review['last_name'], 0, 1) . '.'); ?>
                                        </span>
                                        <?php if ($review['is_verified']): ?>
                                            <span class="ml-2 bg-green-100 text-green-800 text-xs px-2 py-1 rounded-full">Verified Purchase</span>
                                        <?php endif; ?>
                                    </div>
                                    <span class="text-sm text-gray-500"><?php echo date('M j, Y', strtotime($review['created_at'])); ?></span>
                                </div>
                                
                                <p class="text-gray-700"><?php echo nl2br(htmlspecialchars($review['review_text'])); ?></p>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</section>

<!-- Related Products -->
<?php if (!empty($related_products)): ?>
<section class="py-16 bg-white">
    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <h2 class="text-2xl font-bold text-gray-900 mb-8">Related Products</h2>
        
        <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
            <?php foreach ($related_products as $related): ?>
            <div class="bg-white rounded-lg shadow-sm hover:shadow-lg transition-shadow overflow-hidden">
                <div class="aspect-w-16 aspect-h-12 bg-gray-100">
                    <img src="<?php echo htmlspecialchars($related['image_url']); ?>" 
                         alt="<?php echo htmlspecialchars($related['name']); ?>"
                         class="w-full h-48 object-cover">
                </div>
                <div class="p-4">
                    <h3 class="text-lg font-semibold text-gray-900 mb-2 line-clamp-2">
                        <a href="product.php?id=<?php echo $related['id']; ?>" class="hover:text-primary-600">
                            <?php echo htmlspecialchars($related['name']); ?>
                        </a>
                    </h3>
                    
                    <div class="flex items-center mb-3">
                        <div class="flex items-center">
                            <?php for ($i = 1; $i <= 5; $i++): ?>
                                <svg class="w-4 h-4 <?php echo $i <= floor($related['average_rating'] ?? 0) ? 'text-yellow-400' : 'text-gray-300'; ?>" 
                                     fill="currentColor" viewBox="0 0 20 20">
                                    <path d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.921-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.197-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z"></path>
                                </svg>
                            <?php endfor; ?>
                            <span class="ml-2 text-sm text-gray-600">(<?php echo $related['review_count'] ?? 0; ?>)</span>
                        </div>
                    </div>
                    
                    <div class="flex items-center justify-between">
                        <div class="flex items-center space-x-2">
                            <?php if (!empty($related['sale_price'])): ?>
                                <span class="text-lg font-bold text-primary-600">₦<?php echo number_format($related['sale_price'], 0); ?></span>
                                <span class="text-sm text-gray-500 line-through">₦<?php echo number_format($related['price'], 0); ?></span>
                            <?php else: ?>
                                <span class="text-lg font-bold text-gray-900">₦<?php echo number_format($related['price'], 0); ?></span>
                            <?php endif; ?>
                        </div>
                        <a href="product.php?id=<?php echo $related['id']; ?>" 
                           class="bg-primary-500 text-white px-3 py-2 rounded-md hover:bg-primary-600 text-sm font-medium">
                            View
                        </a>
                    </div>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
    </div>
</section>
<?php endif; ?>

<script>
// Product detail page functionality
function changeMainImage(imageUrl) {
    document.getElementById('main-image').src = imageUrl;
    
    // Update active thumbnail
    const thumbnails = document.querySelectorAll('button[onclick*="changeMainImage"]');
    thumbnails.forEach(thumb => {
        thumb.classList.remove('border-primary-500');
        thumb.classList.add('border-transparent');
    });
    
    event.target.closest('button').classList.remove('border-transparent');
    event.target.closest('button').classList.add('border-primary-500');
}

function showTab(tabName) {
    // Hide all tab contents
    const contents = document.querySelectorAll('.tab-content');
    contents.forEach(content => content.classList.add('hidden'));
    
    // Remove active state from all tabs
    const tabs = document.querySelectorAll('.tab-button');
    tabs.forEach(tab => {
        tab.classList.remove('border-primary-500', 'text-primary-600');
        tab.classList.add('border-transparent', 'text-gray-500');
    });
    
    // Show selected tab content
    document.getElementById(tabName + '-content').classList.remove('hidden');
    
    // Add active state to selected tab
    const activeTab = document.getElementById(tabName + '-tab');
    activeTab.classList.remove('border-transparent', 'text-gray-500');
    activeTab.classList.add('border-primary-500', 'text-primary-600');
}

function handleAddToCart(event) {
    event.preventDefault();
    const quantity = document.getElementById('quantity').value;
    addToCart(<?php echo $product['id']; ?>, parseInt(quantity));
}

function addToWishlist(productId) {
    // Check if user is logged in
    fetch('api/check_auth.php')
        .then(response => response.json())
        .then(data => {
            if (!data.authenticated) {
                window.location.href = 'login.php?redirect=' + encodeURIComponent(window.location.href);
                return;
            }
            
            // Add to wishlist logic would go here
            showNotification('Added to wishlist!', 'success');
        })
        .catch(error => {
            console.error('Error:', error);
            showNotification('An error occurred. Please try again.', 'error');
        });
}
</script>

<?php include 'includes/footer.php'; ?>
</merged_code>
