<?php
$page_title = "Products";
$page_description = "Browse our complete collection of premium smartphones, household appliances, and electronics with competitive prices and fast shipping.";

require_once 'config/database.php';
include 'includes/header.php';

$conn = getDBConnection();

// Get filter parameters
$category_filter = isset($_GET['category']) ? $_GET['category'] : '';
$search_query = isset($_GET['q']) ? trim($_GET['q']) : '';
$sort_by = isset($_GET['sort']) ? $_GET['sort'] : 'newest';
$min_price = isset($_GET['min_price']) ? floatval($_GET['min_price']) : 0;
$max_price = isset($_GET['max_price']) ? floatval($_GET['max_price']) : 0;
$brand_filter = isset($_GET['brand']) ? $_GET['brand'] : '';

// Pagination
$page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
$per_page = 12;
$offset = ($page - 1) * $per_page;

// Build WHERE clause
$where_conditions = ["p.is_active = 1"];
$params = [];
$param_types = "";

if ($category_filter) {
    $where_conditions[] = "c.slug = ?";
    $params[] = $category_filter;
    $param_types .= "s";
}

if ($search_query) {
    $where_conditions[] = "(p.name LIKE ? OR p.description LIKE ? OR p.short_description LIKE ?)";
    $search_param = "%$search_query%";
    $params[] = $search_param;
    $params[] = $search_param;
    $params[] = $search_param;
    $param_types .= "sss";
}

if ($min_price > 0) {
    $where_conditions[] = "COALESCE(p.sale_price, p.price) >= ?";
    $params[] = $min_price;
    $param_types .= "d";
}

if ($max_price > 0) {
    $where_conditions[] = "COALESCE(p.sale_price, p.price) <= ?";
    $params[] = $max_price;
    $param_types .= "d";
}

if ($brand_filter) {
    $where_conditions[] = "p.brand = ?";
    $params[] = $brand_filter;
    $param_types .= "s";
}

$where_clause = implode(" AND ", $where_conditions);

// Build ORDER BY clause
$order_clause = "p.created_at DESC";
switch ($sort_by) {
    case 'price_low':
        $order_clause = "COALESCE(p.sale_price, p.price) ASC";
        break;
    case 'price_high':
        $order_clause = "COALESCE(p.sale_price, p.price) DESC";
        break;
    case 'rating':
        $order_clause = "p.rating DESC, p.review_count DESC";
        break;
    case 'name':
        $order_clause = "p.name ASC";
        break;
    case 'newest':
    default:
        $order_clause = "p.created_at DESC";
        break;
}

// Get total count for pagination
$count_sql = "SELECT COUNT(*) as total 
              FROM products p 
              LEFT JOIN categories c ON p.category_id = c.id 
              WHERE $where_clause";

$count_stmt = $conn->prepare($count_sql);
if (!empty($params)) {
    $count_stmt->bind_param($param_types, ...$params);
}
$count_stmt->execute();
$total_products = $count_stmt->get_result()->fetch_assoc()['total'];
$total_pages = ceil($total_products / $per_page);

// Get products
$sql = "SELECT p.*, c.name as category_name, c.slug as category_slug
        FROM products p 
        LEFT JOIN categories c ON p.category_id = c.id 
        WHERE $where_clause 
        ORDER BY $order_clause 
        LIMIT ? OFFSET ?";

$stmt = $conn->prepare($sql);
$params[] = $per_page;
$params[] = $offset;
$param_types .= "ii";
$stmt->bind_param($param_types, ...$params);
$stmt->execute();
$products = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);

// Get categories for filter
$categories = [];
$cat_result = $conn->query("SELECT * FROM categories WHERE is_active = 1 ORDER BY name ASC");
if ($cat_result) {
    $categories = $cat_result->fetch_all(MYSQLI_ASSOC);
}

// Get brands for filter
$brands = [];
$brand_result = $conn->query("SELECT DISTINCT brand FROM products WHERE is_active = 1 AND brand IS NOT NULL ORDER BY brand ASC");
if ($brand_result) {
    while ($row = $brand_result->fetch_assoc()) {
        $brands[] = $row['brand'];
    }
}

$conn->close();

// Build current URL for filters
function buildFilterUrl($new_params = []) {
    $current_params = $_GET;
    $current_params = array_merge($current_params, $new_params);
    
    // Remove empty parameters
    $current_params = array_filter($current_params, function($value) {
        return $value !== '' && $value !== null;
    });
    
    // Remove page when changing filters
    if (!isset($new_params['page']) && count($new_params) > 0) {
        unset($current_params['page']);
    }
    
    return 'products.php' . (empty($current_params) ? '' : '?' . http_build_query($current_params));
}
?>

<!-- Page Header -->
<section class="bg-gray-100 py-8">
    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <div class="flex flex-col md:flex-row md:items-center md:justify-between">
            <div>
                <h1 class="text-3xl font-bold text-gray-900">
                    <?php if ($category_filter): ?>
                        <?php 
                        $cat_name = '';
                        foreach ($categories as $cat) {
                            if ($cat['slug'] === $category_filter) {
                                $cat_name = $cat['name'];
                                break;
                            }
                        }
                        echo htmlspecialchars($cat_name);
                        ?>
                    <?php elseif ($search_query): ?>
                        Search Results for "<?php echo htmlspecialchars($search_query); ?>"
                    <?php else: ?>
                        All Products
                    <?php endif; ?>
                </h1>
                <p class="text-gray-600 mt-2">
                    Showing <?php echo count($products); ?> of <?php echo $total_products; ?> products
                </p>
            </div>
            
            <!-- Sort Dropdown -->
            <div class="mt-4 md:mt-0">
                <select onchange="window.location.href=this.value" 
                        class="border border-gray-300 rounded-lg px-4 py-2 bg-white focus:ring-2 focus:ring-primary-500 focus:border-transparent">
                    <option value="<?php echo buildFilterUrl(['sort' => 'newest']); ?>" <?php echo $sort_by === 'newest' ? 'selected' : ''; ?>>
                        Newest First
                    </option>
                    <option value="<?php echo buildFilterUrl(['sort' => 'price_low']); ?>" <?php echo $sort_by === 'price_low' ? 'selected' : ''; ?>>
                        Price: Low to High
                    </option>
                    <option value="<?php echo buildFilterUrl(['sort' => 'price_high']); ?>" <?php echo $sort_by === 'price_high' ? 'selected' : ''; ?>>
                        Price: High to Low
                    </option>
                    <option value="<?php echo buildFilterUrl(['sort' => 'rating']); ?>" <?php echo $sort_by === 'rating' ? 'selected' : ''; ?>>
                        Highest Rated
                    </option>
                    <option value="<?php echo buildFilterUrl(['sort' => 'name']); ?>" <?php echo $sort_by === 'name' ? 'selected' : ''; ?>>
                        Name A-Z
                    </option>
                </select>
            </div>
        </div>
    </div>
</section>

<!-- Main Content -->
<section class="py-8">
    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <div class="flex flex-col lg:flex-row gap-8">
            <!-- Filters Sidebar -->
            <div class="lg:w-1/4">
                <div class="bg-white rounded-lg shadow-sm p-6 sticky top-24">
                    <h3 class="text-lg font-semibold text-gray-900 mb-4">Filters</h3>
                    
                    <!-- Categories Filter -->
                    <div class="mb-6">
                        <h4 class="font-medium text-gray-900 mb-3">Categories</h4>
                        <div class="space-y-2">
                            <a href="<?php echo buildFilterUrl(['category' => '']); ?>" 
                               class="block text-sm <?php echo !$category_filter ? 'text-primary-600 font-medium' : 'text-gray-600 hover:text-primary-600'; ?>">
                                All Categories
                            </a>
                            <?php foreach ($categories as $category): ?>
                            <a href="<?php echo buildFilterUrl(['category' => $category['slug']]); ?>" 
                               class="block text-sm <?php echo $category_filter === $category['slug'] ? 'text-primary-600 font-medium' : 'text-gray-600 hover:text-primary-600'; ?>">
                                <?php echo htmlspecialchars($category['name']); ?>
                            </a>
                            <?php endforeach; ?>
                        </div>
                    </div>
                    
                    <!-- Brand Filter -->
                    <?php if (!empty($brands)): ?>
                    <div class="mb-6">
                        <h4 class="font-medium text-gray-900 mb-3">Brands</h4>
                        <div class="space-y-2">
                            <a href="<?php echo buildFilterUrl(['brand' => '']); ?>" 
                               class="block text-sm <?php echo !$brand_filter ? 'text-primary-600 font-medium' : 'text-gray-600 hover:text-primary-600'; ?>">
                                All Brands
                            </a>
                            <?php foreach ($brands as $brand): ?>
                            <a href="<?php echo buildFilterUrl(['brand' => $brand]); ?>" 
                               class="block text-sm <?php echo $brand_filter === $brand ? 'text-primary-600 font-medium' : 'text-gray-600 hover:text-primary-600'; ?>">
                                <?php echo htmlspecialchars($brand); ?>
                            </a>
                            <?php endforeach; ?>
                        </div>
                    </div>
                    <?php endif; ?>
                    
                    <!-- Price Range Filter -->
                    <div class="mb-6">
                        <h4 class="font-medium text-gray-900 mb-3">Price Range</h4>
                        <form method="GET" class="space-y-3">
                            <!-- Preserve other filters -->
                            <?php foreach ($_GET as $key => $value): ?>
                                <?php if (!in_array($key, ['min_price', 'max_price', 'page'])): ?>
                                    <input type="hidden" name="<?php echo htmlspecialchars($key); ?>" value="<?php echo htmlspecialchars($value); ?>">
                                <?php endif; ?>
                            <?php endforeach; ?>
                            
                            <div class="flex space-x-2">
                                <input type="number" name="min_price" placeholder="Min" 
                                       value="<?php echo $min_price > 0 ? $min_price : ''; ?>"
                                       class="w-full px-3 py-2 border border-gray-300 rounded-md text-sm focus:ring-2 focus:ring-primary-500 focus:border-transparent">
                                <input type="number" name="max_price" placeholder="Max" 
                                       value="<?php echo $max_price > 0 ? $max_price : ''; ?>"
                                       class="w-full px-3 py-2 border border-gray-300 rounded-md text-sm focus:ring-2 focus:ring-primary-500 focus:border-transparent">
                            </div>
                            <button type="submit" class="w-full bg-primary-500 text-white py-2 rounded-md hover:bg-primary-600 text-sm font-medium">
                                Apply
                            </button>
                        </form>
                        
                        <!-- Quick price filters -->
                        <div class="mt-3 space-y-1">
                            <a href="<?php echo buildFilterUrl(['min_price' => '', 'max_price' => 80000]); ?>" 
                               class="block text-sm text-gray-600 hover:text-primary-600">Under ₦80,000</a>
                            <a href="<?php echo buildFilterUrl(['min_price' => 80000, 'max_price' => 400000]); ?>" 
                               class="block text-sm text-gray-600 hover:text-primary-600">₦80,000 - ₦400,000</a>
                            <a href="<?php echo buildFilterUrl(['min_price' => 400000, 'max_price' => 800000]); ?>" 
                               class="block text-sm text-gray-600 hover:text-primary-600">₦400,000 - ₦800,000</a>
                            <a href="<?php echo buildFilterUrl(['min_price' => 800000, 'max_price' => '']); ?>" 
                               class="block text-sm text-gray-600 hover:text-primary-600">Over ₦800,000</a>
                        </div>
                    </div>
                    
                    <!-- Clear Filters -->
                    <?php if ($category_filter || $search_query || $brand_filter || $min_price > 0 || $max_price > 0): ?>
                    <div>
                        <a href="products.php" class="text-sm text-gray-600 hover:text-primary-600 underline">
                            Clear All Filters
                        </a>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
            
            <!-- Products Grid -->
            <div class="lg:w-3/4">
                <?php if (empty($products)): ?>
                    <!-- No Products Found -->
                    <div class="text-center py-12">
                        <svg class="w-16 h-16 text-gray-400 mx-auto mb-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M20 13V6a2 2 0 00-2-2H6a2 2 0 00-2 2v7m16 0v5a2 2 0 01-2 2H6a2 2 0 01-2-2v-5m16 0h-2M4 13h2m13-8l-2-2m0 0l-2 2m2-2v6"></path>
                        </svg>
                        <h3 class="text-xl font-semibold text-gray-900 mb-2">No products found</h3>
                        <p class="text-gray-600 mb-4">Try adjusting your filters or search terms.</p>
                        <a href="products.php" class="bg-primary-500 text-white px-6 py-2 rounded-lg hover:bg-primary-600 font-medium">
                            View All Products
                        </a>
                    </div>
                <?php else: ?>
                    <!-- Products Grid -->
                    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                        <?php foreach ($products as $product): ?>
                        <div class="bg-white rounded-lg shadow-sm hover:shadow-lg transition-shadow overflow-hidden">
                            <div class="aspect-w-16 aspect-h-12 bg-gray-100">
                                <img src="<?php echo htmlspecialchars($product['image_url']); ?>" 
                                     alt="<?php echo htmlspecialchars($product['name']); ?>"
                                     class="w-full h-48 object-cover">
                            </div>
                            <div class="p-4">
                                <div class="flex items-start justify-between mb-2">
                                    <h3 class="text-lg font-semibold text-gray-900 line-clamp-2">
                                        <a href="product.php?id=<?php echo $product['id']; ?>" class="hover:text-primary-600">
                                            <?php echo htmlspecialchars($product['name']); ?>
                                        </a>
                                    </h3>
                                    <?php if ($product['is_featured']): ?>
                                        <span class="bg-primary-100 text-primary-800 text-xs px-2 py-1 rounded-full font-medium">Featured</span>
                                    <?php endif; ?>
                                </div>
                                
                                <p class="text-sm text-gray-600 mb-2">
                                    <?php echo htmlspecialchars($product['brand']); ?> • <?php echo htmlspecialchars($product['category_name']); ?>
                                </p>
                                
                                <p class="text-gray-600 text-sm mb-3 line-clamp-2">
                                    <?php echo htmlspecialchars($product['short_description']); ?>
                                </p>
                                
                                <!-- Rating -->
                                <div class="flex items-center mb-3">
                                    <div class="flex items-center">
                                        <?php for ($i = 1; $i <= 5; $i++): ?>
                                            <svg class="w-4 h-4 <?php echo $i <= floor($product['average_rating'] ?? 0) ? 'text-yellow-400' : 'text-gray-300'; ?>" 
                                                 fill="currentColor" viewBox="0 0 20 20">
                                                <path d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.921-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.197-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z"></path>
                                            </svg>
                                        <?php endfor; ?>
                                        <span class="ml-2 text-sm text-gray-600">(<?php echo $product['review_count'] ?? 0; ?>)</span>
                                    </div>
                                </div>
                                
                                <!-- Price and Actions -->
                                <div class="space-y-3">
                                    <!-- Price -->
                                    <div class="flex items-center space-x-2">
                                        <?php if (!empty($product['sale_price'])): ?>
                                            <span class="text-xl font-bold text-primary-600">₦<?php echo number_format($product['sale_price'], 0); ?></span>
                                            <span class="text-sm text-gray-500 line-through">₦<?php echo number_format($product['price'], 0); ?></span>
                                        <?php else: ?>
                                            <span class="text-xl font-bold text-gray-900">₦<?php echo number_format($product['price'], 0); ?></span>
                                        <?php endif; ?>
                                    </div>
                                    
                                    <!-- Buttons -->
                                    <div class="flex space-x-2">
                                        <button onclick="addToCart(<?php echo $product['id']; ?>)" 
                                                class="flex-1 bg-primary-500 text-white px-3 py-2 rounded-md hover:bg-primary-600 text-sm font-medium">
                                            Add to Cart
                                        </button>
                                        <a href="product.php?id=<?php echo $product['id']; ?>" 
                                           class="flex-1 border border-gray-300 text-gray-700 px-3 py-2 rounded-md hover:bg-gray-50 text-sm font-medium text-center">
                                            View
                                        </a>
                                    </div>
                                </div>
                                
                                <!-- Stock Status -->
                                <?php if ($product['quantity'] <= 0): ?>
                                    <div class="mt-2">
                                        <span class="text-red-600 text-sm font-medium">Out of Stock</span>
                                    </div>
                                <?php elseif ($product['quantity'] <= 5): ?>
                                    <div class="mt-2">
                                        <span class="text-orange-600 text-sm font-medium">Only <?php echo $product['quantity']; ?> left!</span>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    </div>
                    
                    <!-- Pagination -->
                    <?php if ($total_pages > 1): ?>
                    <div class="mt-8 flex justify-center">
                        <nav class="flex items-center space-x-2">
                            <!-- Previous Page -->
                            <?php if ($page > 1): ?>
                                <a href="<?php echo buildFilterUrl(['page' => $page - 1]); ?>" 
                                   class="px-3 py-2 border border-gray-300 rounded-md text-gray-700 hover:bg-gray-50">
                                    Previous
                                </a>
                            <?php endif; ?>
                            
                            <!-- Page Numbers -->
                            <?php
                            $start_page = max(1, $page - 2);
                            $end_page = min($total_pages, $page + 2);
                            
                            if ($start_page > 1): ?>
                                <a href="<?php echo buildFilterUrl(['page' => 1]); ?>" 
                                   class="px-3 py-2 border border-gray-300 rounded-md text-gray-700 hover:bg-gray-50">1</a>
                                <?php if ($start_page > 2): ?>
                                    <span class="px-3 py-2 text-gray-500">...</span>
                                <?php endif; ?>
                            <?php endif; ?>
                            
                            <?php for ($i = $start_page; $i <= $end_page; $i++): ?>
                                <?php if ($i == $page): ?>
                                    <span class="px-3 py-2 bg-primary-500 text-white rounded-md"><?php echo $i; ?></span>
                                <?php else: ?>
                                    <a href="<?php echo buildFilterUrl(['page' => $i]); ?>" 
                                       class="px-3 py-2 border border-gray-300 rounded-md text-gray-700 hover:bg-gray-50"><?php echo $i; ?></a>
                                <?php endif; ?>
                            <?php endfor; ?>
                            
                            <?php if ($end_page < $total_pages): ?>
                                <?php if ($end_page < $total_pages - 1): ?>
                                    <span class="px-3 py-2 text-gray-500">...</span>
                                <?php endif; ?>
                                <a href="<?php echo buildFilterUrl(['page' => $total_pages]); ?>" 
                                   class="px-3 py-2 border border-gray-300 rounded-md text-gray-700 hover:bg-gray-50"><?php echo $total_pages; ?></a>
                            <?php endif; ?>
                            
                            <!-- Next Page -->
                            <?php if ($page < $total_pages): ?>
                                <a href="<?php echo buildFilterUrl(['page' => $page + 1]); ?>" 
                                   class="px-3 py-2 border border-gray-300 rounded-md text-gray-700 hover:bg-gray-50">
                                    Next
                                </a>
                            <?php endif; ?>
                        </nav>
                    </div>
                    <?php endif; ?>
                <?php endif; ?>
            </div>
        </div>
    </div>
</section>

<?php include 'includes/footer.php'; ?>

</merged_code>
